import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import Layout from "../components/Layout";
import ImportDialog from "../components/ImportDialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from "@/components/ui/alert-dialog";
import { useToast } from "@/components/ui/use-toast";
import { Plus, Search, Edit, Trash2, Key, RotateCcw, Eye, EyeOff } from "lucide-react";
import backend from "~backend/client";
import type { CreateMadrasahRequest } from "~backend/madrasah/create_madrasah";

interface Madrasah {
  id: string;
  nsm: string;
  npsn: string;
  nama_madrasah: string;
  alamat: string;
  kecamatan: string;
  kabupaten: string;
  provinsi: string;
  akreditasi: string;
  status: string;
  nama_kepala: string;
  operator_token?: string;
  operator_registered?: boolean;
}

export default function MadrasahPage() {
  const [search, setSearch] = useState("");
  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [editingMadrasah, setEditingMadrasah] = useState<Madrasah | null>(null);
  const [showTokenDialog, setShowTokenDialog] = useState(false);
  const [selectedMadrasah, setSelectedMadrasah] = useState<Madrasah | null>(null);
  const [showPassword, setShowPassword] = useState<{ [key: string]: boolean }>({});
  const [formData, setFormData] = useState<CreateMadrasahRequest>({
    nsm: "",
    npsn: "",
    nama_madrasah: "",
    alamat: "",
    kecamatan: "",
    kabupaten: "",
    provinsi: "",
    akreditasi: "",
    status: "",
    nama_kepala: "",
  });

  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data, isLoading } = useQuery({
    queryKey: ["madrasah", search],
    queryFn: () => backend.madrasah.listMadrasah({ search, limit: 100 }),
  });

  const createMutation = useMutation({
    mutationFn: (data: CreateMadrasahRequest) => backend.madrasah.createMadrasah(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["madrasah"] });
      setIsCreateDialogOpen(false);
      resetForm();
      toast({
        title: "Berhasil",
        description: "Data madrasah berhasil ditambahkan",
      });
    },
    onError: (error) => {
      console.error("Create madrasah error:", error);
      toast({
        title: "Gagal",
        description: "Gagal menambahkan data madrasah",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: (data: { id: string } & Partial<CreateMadrasahRequest>) => backend.madrasah.updateMadrasah(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["madrasah"] });
      setIsEditDialogOpen(false);
      setEditingMadrasah(null);
      resetForm();
      toast({
        title: "Berhasil",
        description: "Data madrasah berhasil diperbarui",
      });
    },
    onError: (error) => {
      console.error("Update madrasah error:", error);
      toast({
        title: "Gagal",
        description: "Gagal memperbarui data madrasah",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: string) => backend.madrasah.deleteMadrasah({ id }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["madrasah"] });
      toast({
        title: "Berhasil",
        description: "Data madrasah berhasil dihapus",
      });
    },
    onError: (error) => {
      console.error("Delete madrasah error:", error);
      toast({
        title: "Gagal",
        description: "Gagal menghapus data madrasah",
        variant: "destructive",
      });
    },
  });

  const resetTokenMutation = useMutation({
    mutationFn: (madrasah_id: string) => backend.madrasah.resetToken({ madrasah_id }),
    onSuccess: (result) => {
      queryClient.invalidateQueries({ queryKey: ["madrasah"] });
      toast({
        title: "Token Direset",
        description: `Token baru: ${result.new_token}. Operator harus registrasi ulang.`,
      });
    },
    onError: (error) => {
      console.error("Reset token error:", error);
      toast({
        title: "Gagal",
        description: "Gagal mereset token",
        variant: "destructive",
      });
    },
  });

  const importMutation = useMutation({
    mutationFn: (data: any[]) => {
      const mappedData = data.map((row) => ({
        nsm: String(row["NSM"] || row["nsm"] || "").trim(),
        npsn: String(row["NPSN"] || row["npsn"] || "").trim(),
        nama_madrasah: String(row["Nama Madrasah"] || row["nama_madrasah"] || "").trim(),
        alamat: String(row["Alamat"] || row["alamat"] || "").trim(),
        kecamatan: String(row["Kecamatan"] || row["kecamatan"] || "").trim(),
        kabupaten: String(row["Kabupaten"] || row["kabupaten"] || "").trim(),
        provinsi: String(row["Provinsi"] || row["provinsi"] || "").trim(),
        akreditasi: String(row["Akreditasi"] || row["akreditasi"] || "").trim(),
        status: String(row["Status"] || row["status"] || "").trim(),
        nama_kepala: String(row["Nama Kepala"] || row["nama_kepala"] || "").trim(),
      }));
      
      return backend.madrasah.importMadrasah({ data: mappedData });
    },
    onSuccess: (result) => {
      queryClient.invalidateQueries({ queryKey: ["madrasah"] });
      toast({
        title: "Import Selesai",
        description: `Berhasil: ${result.success}, Gagal: ${result.failed}`,
      });
      if (result.errors.length > 0) {
        console.error("Import errors:", result.errors);
      }
    },
    onError: (error) => {
      console.error("Import madrasah error:", error);
      toast({
        title: "Import Gagal",
        description: "Gagal mengimport data madrasah",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      nsm: "",
      npsn: "",
      nama_madrasah: "",
      alamat: "",
      kecamatan: "",
      kabupaten: "",
      provinsi: "",
      akreditasi: "",
      status: "",
      nama_kepala: "",
    });
  };

  const handleCreate = (e: React.FormEvent) => {
    e.preventDefault();
    createMutation.mutate(formData);
  };

  const handleEdit = (madrasah: Madrasah) => {
    setEditingMadrasah(madrasah);
    setFormData({
      nsm: madrasah.nsm,
      npsn: madrasah.npsn,
      nama_madrasah: madrasah.nama_madrasah,
      alamat: madrasah.alamat,
      kecamatan: madrasah.kecamatan,
      kabupaten: madrasah.kabupaten,
      provinsi: madrasah.provinsi,
      akreditasi: madrasah.akreditasi,
      status: madrasah.status,
      nama_kepala: madrasah.nama_kepala,
    });
    setIsEditDialogOpen(true);
  };

  const handleUpdate = (e: React.FormEvent) => {
    e.preventDefault();
    if (editingMadrasah) {
      updateMutation.mutate({ id: editingMadrasah.id, ...formData });
    }
  };

  const handleDelete = (id: string) => {
    deleteMutation.mutate(id);
  };

  const handleImport = async (data: any[]) => {
    await importMutation.mutateAsync(data);
  };

  const handleShowToken = (madrasah: Madrasah) => {
    setSelectedMadrasah(madrasah);
    setShowTokenDialog(true);
  };

  const handleResetToken = (madrasahId: string) => {
    resetTokenMutation.mutate(madrasahId);
  };

  const toggleShowPassword = (madrasahId: string) => {
    setShowPassword(prev => ({
      ...prev,
      [madrasahId]: !prev[madrasahId]
    }));
  };

  const templateFields = [
    "NSM",
    "NPSN",
    "Nama Madrasah",
    "Alamat",
    "Kecamatan",
    "Kabupaten",
    "Provinsi",
    "Akreditasi",
    "Status",
    "Nama Kepala"
  ];

  const MadrasahForm = ({ onSubmit, isLoading }: { onSubmit: (e: React.FormEvent) => void; isLoading: boolean }) => (
    <form onSubmit={onSubmit} className="space-y-4">
      <div className="grid grid-cols-2 gap-4">
        <div className="space-y-2">
          <Label htmlFor="nsm">NSM *</Label>
          <Input
            id="nsm"
            value={formData.nsm}
            onChange={(e) => setFormData({ ...formData, nsm: e.target.value })}
            required
          />
        </div>
        <div className="space-y-2">
          <Label htmlFor="npsn">NPSN *</Label>
          <Input
            id="npsn"
            value={formData.npsn}
            onChange={(e) => setFormData({ ...formData, npsn: e.target.value })}
            required
          />
        </div>
      </div>
      <div className="space-y-2">
        <Label htmlFor="nama_madrasah">Nama Madrasah *</Label>
        <Input
          id="nama_madrasah"
          value={formData.nama_madrasah}
          onChange={(e) => setFormData({ ...formData, nama_madrasah: e.target.value })}
          required
        />
      </div>
      <div className="space-y-2">
        <Label htmlFor="alamat">Alamat</Label>
        <Input
          id="alamat"
          value={formData.alamat}
          onChange={(e) => setFormData({ ...formData, alamat: e.target.value })}
        />
      </div>
      <div className="grid grid-cols-3 gap-4">
        <div className="space-y-2">
          <Label htmlFor="kecamatan">Kecamatan</Label>
          <Input
            id="kecamatan"
            value={formData.kecamatan}
            onChange={(e) => setFormData({ ...formData, kecamatan: e.target.value })}
          />
        </div>
        <div className="space-y-2">
          <Label htmlFor="kabupaten">Kabupaten</Label>
          <Input
            id="kabupaten"
            value={formData.kabupaten}
            onChange={(e) => setFormData({ ...formData, kabupaten: e.target.value })}
          />
        </div>
        <div className="space-y-2">
          <Label htmlFor="provinsi">Provinsi</Label>
          <Input
            id="provinsi"
            value={formData.provinsi}
            onChange={(e) => setFormData({ ...formData, provinsi: e.target.value })}
          />
        </div>
      </div>
      <div className="grid grid-cols-3 gap-4">
        <div className="space-y-2">
          <Label htmlFor="akreditasi">Akreditasi</Label>
          <Input
            id="akreditasi"
            value={formData.akreditasi}
            onChange={(e) => setFormData({ ...formData, akreditasi: e.target.value })}
          />
        </div>
        <div className="space-y-2">
          <Label htmlFor="status">Status</Label>
          <Input
            id="status"
            value={formData.status}
            onChange={(e) => setFormData({ ...formData, status: e.target.value })}
          />
        </div>
        <div className="space-y-2">
          <Label htmlFor="nama_kepala">Nama Kepala</Label>
          <Input
            id="nama_kepala"
            value={formData.nama_kepala}
            onChange={(e) => setFormData({ ...formData, nama_kepala: e.target.value })}
          />
        </div>
      </div>
      <div className="flex justify-end space-x-2">
        <Button
          type="button"
          variant="outline"
          onClick={() => {
            setIsCreateDialogOpen(false);
            setIsEditDialogOpen(false);
            resetForm();
          }}
        >
          Batal
        </Button>
        <Button
          type="submit"
          className="bg-teal-600 hover:bg-teal-700"
          disabled={isLoading}
        >
          {isLoading ? "Menyimpan..." : "Simpan"}
        </Button>
      </div>
    </form>
  );

  return (
    <Layout>
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Data Madrasah</h1>
            <p className="text-gray-600 mt-2">Kelola data madrasah di sistem</p>
          </div>
          <div className="flex space-x-2">
            <ImportDialog
              title="Madrasah"
              templateFields={templateFields}
              onImport={handleImport}
              isLoading={importMutation.isPending}
            />
            <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
              <DialogTrigger asChild>
                <Button className="bg-teal-600 hover:bg-teal-700">
                  <Plus className="h-4 w-4 mr-2" />
                  Tambah Madrasah
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-2xl">
                <DialogHeader>
                  <DialogTitle>Tambah Madrasah Baru</DialogTitle>
                </DialogHeader>
                <MadrasahForm onSubmit={handleCreate} isLoading={createMutation.isPending} />
              </DialogContent>
            </Dialog>
          </div>
        </div>

        <Card>
          <CardHeader>
            <div className="flex items-center space-x-2">
              <Search className="h-4 w-4 text-gray-400" />
              <Input
                placeholder="Cari madrasah..."
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                className="max-w-sm"
              />
            </div>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-8">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-teal-600 mx-auto"></div>
                <p className="text-gray-600 mt-2">Memuat data...</p>
              </div>
            ) : (
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b">
                      <th className="text-left py-3 px-4 font-medium text-gray-900">NSM</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Nama Madrasah</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Kecamatan</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">NPSN (Password Admin)</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Token Operator</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Status Operator</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Aksi</th>
                    </tr>
                  </thead>
                  <tbody>
                    {data?.madrasah.map((madrasah) => (
                      <tr key={madrasah.id} className="border-b hover:bg-gray-50">
                        <td className="py-3 px-4 text-sm font-mono">{madrasah.nsm}</td>
                        <td className="py-3 px-4 text-sm font-medium">{madrasah.nama_madrasah}</td>
                        <td className="py-3 px-4 text-sm">{madrasah.kecamatan}</td>
                        <td className="py-3 px-4 text-sm">
                          <div className="flex items-center space-x-2">
                            <code className="bg-gray-100 px-2 py-1 rounded text-xs font-mono">
                              {showPassword[madrasah.id] ? madrasah.npsn : '••••••••'}
                            </code>
                            <Button
                              size="sm"
                              variant="ghost"
                              onClick={() => toggleShowPassword(madrasah.id)}
                              className="h-6 w-6 p-0"
                            >
                              {showPassword[madrasah.id] ? <EyeOff className="h-3 w-3" /> : <Eye className="h-3 w-3" />}
                            </Button>
                          </div>
                        </td>
                        <td className="py-3 px-4 text-sm">
                          <code className="bg-blue-50 text-blue-700 px-2 py-1 rounded text-xs font-mono">
                            {madrasah.operator_token || 'N/A'}
                          </code>
                        </td>
                        <td className="py-3 px-4 text-sm">
                          <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                            madrasah.operator_registered 
                              ? 'bg-green-100 text-green-800' 
                              : 'bg-gray-100 text-gray-800'
                          }`}>
                            {madrasah.operator_registered ? 'Terdaftar' : 'Belum Daftar'}
                          </span>
                        </td>
                        <td className="py-3 px-4 text-sm">
                          <div className="flex space-x-1">
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => handleEdit(madrasah)}
                              className="h-8 w-8 p-0"
                              title="Edit"
                            >
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => handleResetToken(madrasah.id)}
                              className="h-8 w-8 p-0 text-orange-600 hover:text-orange-700"
                              title="Reset Token Operator"
                            >
                              <RotateCcw className="h-4 w-4" />
                            </Button>
                            <AlertDialog>
                              <AlertDialogTrigger asChild>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-8 w-8 p-0 text-red-600 hover:text-red-700 hover:bg-red-50"
                                  title="Hapus"
                                >
                                  <Trash2 className="h-4 w-4" />
                                </Button>
                              </AlertDialogTrigger>
                              <AlertDialogContent>
                                <AlertDialogHeader>
                                  <AlertDialogTitle>Hapus Madrasah</AlertDialogTitle>
                                  <AlertDialogDescription>
                                    Apakah Anda yakin ingin menghapus madrasah "{madrasah.nama_madrasah}"? 
                                    Tindakan ini akan menghapus semua data terkait (guru, siswa, karyawan, alumni) dan tidak dapat dibatalkan.
                                  </AlertDialogDescription>
                                </AlertDialogHeader>
                                <AlertDialogFooter>
                                  <AlertDialogCancel>Batal</AlertDialogCancel>
                                  <AlertDialogAction
                                    onClick={() => handleDelete(madrasah.id)}
                                    className="bg-red-600 hover:bg-red-700"
                                  >
                                    Hapus
                                  </AlertDialogAction>
                                </AlertDialogFooter>
                              </AlertDialogContent>
                            </AlertDialog>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
                {data?.madrasah.length === 0 && (
                  <div className="text-center py-8 text-gray-600">
                    Tidak ada data madrasah
                  </div>
                )}
              </div>
            )}
          </CardContent>
        </Card>

        <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Edit Madrasah</DialogTitle>
            </DialogHeader>
            <MadrasahForm onSubmit={handleUpdate} isLoading={updateMutation.isPending} />
          </DialogContent>
        </Dialog>
      </div>
    </Layout>
  );
}
