import { useQuery } from "@tanstack/react-query";
import { useAuth } from "../contexts/AuthContext";
import Layout from "../components/Layout";
import RecapDialog from "../components/RecapDialog";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Users, GraduationCap, UserCheck, Award, School, Download } from "lucide-react";
import { Button } from "@/components/ui/button";
import backend from "~backend/client";
import * as XLSX from "xlsx";

export default function DashboardPage() {
  const { user } = useAuth();

  const { data: guruData } = useQuery({
    queryKey: ["guru", user?.madrasahId],
    queryFn: () => backend.guru.listGuru({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined,
      limit: 1 
    }),
  });

  const { data: siswaData } = useQuery({
    queryKey: ["siswa", user?.madrasahId],
    queryFn: () => backend.siswa.listSiswa({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined,
      limit: 1 
    }),
  });

  const { data: karyawanData } = useQuery({
    queryKey: ["karyawan", user?.madrasahId],
    queryFn: () => backend.karyawan.listKaryawan({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined,
      limit: 1 
    }),
  });

  const { data: alumniData } = useQuery({
    queryKey: ["alumni", user?.madrasahId],
    queryFn: () => backend.alumni.listAlumni({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined,
      limit: 1 
    }),
  });

  const { data: madrasahData } = useQuery({
    queryKey: ["madrasah"],
    queryFn: () => backend.madrasah.listMadrasah({ limit: 1 }),
    enabled: user?.userType === "superadmin",
  });

  const { data: recapData } = useQuery({
    queryKey: ["recap", user?.madrasahId],
    queryFn: () => backend.reports.getRecap({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined 
    }),
    enabled: user?.userType === "superadmin",
  });

  const stats = [
    ...(user?.userType === "superadmin" ? [{
      title: "Total Madrasah",
      value: madrasahData?.total || 0,
      icon: School,
      color: "bg-purple-500",
    }] : []),
    {
      title: "Total Guru",
      value: guruData?.total || 0,
      icon: Users,
      color: "bg-blue-500",
    },
    {
      title: "Total Siswa",
      value: siswaData?.total || 0,
      icon: GraduationCap,
      color: "bg-green-500",
    },
    {
      title: "Total Karyawan",
      value: karyawanData?.total || 0,
      icon: UserCheck,
      color: "bg-orange-500",
    },
    {
      title: "Total Alumni",
      value: alumniData?.total || 0,
      icon: Award,
      color: "bg-red-500",
    },
  ];

  const handleDownloadRecap = () => {
    if (!recapData) return;

    const worksheetData = [
      [
        "Nama Madrasah",
        "Total Guru",
        "Total Siswa",
        "Siswa Laki-laki",
        "Siswa Perempuan",
        "Siswa Kelas 1",
        "Siswa Kelas 2",
        "Siswa Kelas 3",
        "Siswa Kelas 4",
        "Siswa Kelas 5",
        "Siswa Kelas 6",
        "Total Alumni",
      ],
      ...recapData.data.map((row) => [
        row.nama_madrasah,
        row.total_guru,
        row.total_siswa,
        row.total_siswa_laki,
        row.total_siswa_perempuan,
        row.total_siswa_kelas_1,
        row.total_siswa_kelas_2,
        row.total_siswa_kelas_3,
        row.total_siswa_kelas_4,
        row.total_siswa_kelas_5,
        row.total_siswa_kelas_6,
        row.total_alumni,
      ]),
      [],
      [
        "TOTAL",
        recapData.grand_total.total_guru,
        recapData.grand_total.total_siswa,
        recapData.grand_total.total_siswa_laki,
        recapData.grand_total.total_siswa_perempuan,
        recapData.grand_total.total_siswa_kelas_1,
        recapData.grand_total.total_siswa_kelas_2,
        recapData.grand_total.total_siswa_kelas_3,
        recapData.grand_total.total_siswa_kelas_4,
        recapData.grand_total.total_siswa_kelas_5,
        recapData.grand_total.total_siswa_kelas_6,
        recapData.grand_total.total_alumni,
      ],
    ];

    const worksheet = XLSX.utils.aoa_to_sheet(worksheetData);
    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, "Rekap Data");

    const date = new Date().toISOString().split("T")[0];
    XLSX.writeFile(workbook, `Rekap_Data_Madrasah_${date}.xlsx`);
  };

  return (
    <Layout>
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Dashboard</h1>
            <p className="text-gray-600 mt-2">
              {user?.userType === "superadmin" 
                ? "Selamat datang, Super Admin" 
                : user?.userType === "admin"
                ? `Selamat datang, Admin ${user?.madrasahName}`
                : `Selamat datang, Operator ${user?.madrasahName}`
              }
            </p>
          </div>
          <RecapDialog />
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {stats.map((stat) => {
            const Icon = stat.icon;
            return (
              <Card key={stat.title}>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium text-gray-600">
                    {stat.title}
                  </CardTitle>
                  <div className={`p-2 rounded-md ${stat.color}`}>
                    <Icon className="h-4 w-4 text-white" />
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold text-gray-900">
                    {stat.value.toLocaleString()}
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {user?.userType === "superadmin" && recapData && (
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle>Rekap Data Per Madrasah</CardTitle>
              <Button onClick={handleDownloadRecap} size="sm">
                <Download className="h-4 w-4 mr-2" />
                Unduh Excel
              </Button>
            </CardHeader>
            <CardContent>
              <div className="overflow-x-auto">
                <table className="w-full text-sm">
                  <thead>
                    <tr className="border-b bg-gray-50">
                      <th className="px-4 py-3 text-left font-medium text-gray-700">Madrasah</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Guru</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Total Siswa</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">L</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">P</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Kls 1</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Kls 2</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Kls 3</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Kls 4</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Kls 5</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Kls 6</th>
                      <th className="px-4 py-3 text-center font-medium text-gray-700">Alumni</th>
                    </tr>
                  </thead>
                  <tbody>
                    {recapData.data.map((row) => (
                      <tr key={row.madrasah_id} className="border-b hover:bg-gray-50">
                        <td className="px-4 py-3 text-gray-900">{row.nama_madrasah}</td>
                        <td className="px-4 py-3 text-center text-gray-700">{row.total_guru}</td>
                        <td className="px-4 py-3 text-center font-medium text-gray-900">{row.total_siswa}</td>
                        <td className="px-4 py-3 text-center text-blue-600">{row.total_siswa_laki}</td>
                        <td className="px-4 py-3 text-center text-pink-600">{row.total_siswa_perempuan}</td>
                        <td className="px-4 py-3 text-center text-gray-700">{row.total_siswa_kelas_1}</td>
                        <td className="px-4 py-3 text-center text-gray-700">{row.total_siswa_kelas_2}</td>
                        <td className="px-4 py-3 text-center text-gray-700">{row.total_siswa_kelas_3}</td>
                        <td className="px-4 py-3 text-center text-gray-700">{row.total_siswa_kelas_4}</td>
                        <td className="px-4 py-3 text-center text-gray-700">{row.total_siswa_kelas_5}</td>
                        <td className="px-4 py-3 text-center text-gray-700">{row.total_siswa_kelas_6}</td>
                        <td className="px-4 py-3 text-center text-gray-700">{row.total_alumni}</td>
                      </tr>
                    ))}
                    <tr className="border-t-2 bg-gray-100 font-bold">
                      <td className="px-4 py-3 text-gray-900">TOTAL</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_guru}</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_siswa}</td>
                      <td className="px-4 py-3 text-center text-blue-600">{recapData.grand_total.total_siswa_laki}</td>
                      <td className="px-4 py-3 text-center text-pink-600">{recapData.grand_total.total_siswa_perempuan}</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_siswa_kelas_1}</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_siswa_kelas_2}</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_siswa_kelas_3}</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_siswa_kelas_4}</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_siswa_kelas_5}</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_siswa_kelas_6}</td>
                      <td className="px-4 py-3 text-center text-gray-900">{recapData.grand_total.total_alumni}</td>
                    </tr>
                  </tbody>
                </table>
              </div>
            </CardContent>
          </Card>
        )}

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <Card>
            <CardHeader>
              <CardTitle>Informasi Sistem</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex justify-between">
                <span className="text-gray-600">Versi Aplikasi:</span>
                <span className="font-medium">1.0.0</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Status:</span>
                <span className="font-medium text-green-600">Aktif</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Tipe Pengguna:</span>
                <span className="font-medium capitalize">
                  {user?.userType === "superadmin" ? "Super Admin" : user?.userType === "admin" ? "Admin" : "Operator"}
                </span>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Aktivitas Terbaru</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <div className="flex items-center space-x-3">
                  <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                  <span className="text-sm text-gray-600">Sistem beroperasi normal</span>
                </div>
                <div className="flex items-center space-x-3">
                  <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
                  <span className="text-sm text-gray-600">Database tersinkronisasi</span>
                </div>
                <div className="flex items-center space-x-3">
                  <div className="w-2 h-2 bg-yellow-500 rounded-full"></div>
                  <span className="text-sm text-gray-600">Backup otomatis aktif</span>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </Layout>
  );
}
