import { api, APIError } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface ImportSiswaRequest {
  madrasah_id: string;
  data: SiswaImportData[];
}

interface SiswaImportData {
  nisn?: string;
  nis?: string;
  nama_lengkap: string;
  tempat_lahir?: string;
  tanggal_lahir?: string;
  jenis_kelamin?: string;
  agama?: string;
  alamat?: string;
  no_telepon?: string;
  nama_ayah?: string;
  nama_ibu?: string;
  pekerjaan_ayah?: string;
  pekerjaan_ibu?: string;
  kelas?: string;
  tahun_masuk?: number;
  status?: string;
}

interface ImportResult {
  success: number;
  failed: number;
  errors: string[];
}

// Imports multiple siswa records
export const importSiswa = api<ImportSiswaRequest, ImportResult>(
  { expose: true, method: "POST", path: "/siswa/import" },
  async (req) => {
    const { madrasah_id, data } = req;
    
    let success = 0;
    let failed = 0;
    const errors: string[] = [];
    
    console.log("Import siswa request received:", { madrasah_id, dataCount: data.length });
    
    for (let i = 0; i < data.length; i++) {
      const siswa = data[i];
      try {
        console.log(`Processing siswa ${i + 1}:`, siswa);
        
        if (!siswa.nama_lengkap || siswa.nama_lengkap.trim() === "") {
          throw new Error("Nama lengkap is required");
        }
        
        // Parse tanggal_lahir if provided
        let tanggalLahir = null;
        if (siswa.tanggal_lahir && siswa.tanggal_lahir.trim() !== "") {
          const dateStr = siswa.tanggal_lahir.trim();
          const date = new Date(dateStr);
          if (!isNaN(date.getTime())) {
            tanggalLahir = date.toISOString().split('T')[0];
          }
        }
        
        // Parse tahun_masuk
        let tahunMasuk = null;
        if (siswa.tahun_masuk) {
          const year = parseInt(String(siswa.tahun_masuk));
          if (!isNaN(year) && year > 1900 && year < 2100) {
            tahunMasuk = year;
          }
        }
        
        await db.exec`
          INSERT INTO siswa (madrasah_id, nisn, nis, nama_lengkap, tempat_lahir, tanggal_lahir, jenis_kelamin, agama, alamat, 
                            no_telepon, nama_ayah, nama_ibu, pekerjaan_ayah, pekerjaan_ibu, kelas, tahun_masuk, status)
          VALUES (${madrasah_id}, ${(siswa.nisn || "").trim()}, ${(siswa.nis || "").trim()}, ${siswa.nama_lengkap.trim()}, 
                  ${(siswa.tempat_lahir || "").trim()}, ${tanggalLahir}, ${(siswa.jenis_kelamin || "").trim()}, 
                  ${(siswa.agama || "").trim()}, ${(siswa.alamat || "").trim()}, ${(siswa.no_telepon || "").trim()}, 
                  ${(siswa.nama_ayah || "").trim()}, ${(siswa.nama_ibu || "").trim()}, ${(siswa.pekerjaan_ayah || "").trim()}, 
                  ${(siswa.pekerjaan_ibu || "").trim()}, ${(siswa.kelas || "").trim()}, ${tahunMasuk}, 
                  ${(siswa.status || "aktif").trim()})
        `;
        success++;
        console.log(`Successfully imported siswa ${i + 1}`);
      } catch (error) {
        failed++;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        errors.push(`Row ${i + 1}: ${errorMessage}`);
        console.error(`Failed to import siswa ${i + 1}:`, errorMessage);
      }
    }
    
    console.log(`Import siswa completed. Success: ${success}, Failed: ${failed}`);
    return { success, failed, errors };
  }
);
